SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
/* 
====================================================================================================
Author:			luis.ferrao@cbsi.com
Create date:	23.4.2012
Description:	Formats all CNET hosted content as XML. The hierarchy reflects three
				main document types: images, xml and documents. This XML can be transformed into 
				a JSON document for browsers to intern using the CCS_DCXml2JSON.xsl transform
Sample output:	EXEC CCS_sp_get_DC2XML '4452711' yields
				<cnet>
				  <images type="Product shot">
					<imageCollection image-id="b2e1300f-1ab2-430f-bf93-037cda7602a1" media-type="15" angle="Right-angle">
					  <image mime-type="image/jpeg" url="http://cdn.cnetcontent.com/00/b6/00b629d3-fe91-4c8f-96ba-ab43e1f17832.jpg">
						<attribute atrId="1" name="Image Angle" value="Right-angle" />
						<attribute atrId="2" name="Image Type" value="Product shot" />
						<attribute atrId="3" name="Image Width" value="75" />
						<attribute atrId="4" name="Image Height" value="75" />
						<attribute atrId="5" name="File Size" value="1736" />
						<attribute atrId="6" name="Resolution" value="75 x 75" />
						<attribute atrId="7" name="Image Weight" value="497" />
						<attribute atrId="8" name="Image ID" value="b2e1300f-1ab2-430f-bf93-037cda7602a1" />
					  </image>
					  .....
					</imageCollection>
				  </images>
				  <xmlCollection>
					<xml type="5" description="Key Selling Points" url="http://cdn.cnetcontent.com/bd/78/bd78e650-1b65-4ed1-9515-e90adb9a2139.xml" />
				  </xmlCollection>
				  <docCollection>
					<doc type="11" description="Product Data Sheet / Brochure" url="http://cdn.cnetcontent.com/3a/4f/3a4f99fe-4e1e-4038-914a-89ef4ef951d2.pdf" />
					<doc type="12" description="User Manual" url="http://cdn.cnetcontent.com/9e/bb/9ebb1688-82ab-4d1b-8dfc-353e7484a89f.pdf" />
					<doc type="13" description="Quick Start Guide" url="http://cdn.cnetcontent.com/e0/96/e096644d-ba30-4e24-8e3b-974ae6da4d21.pdf" />
					<doc type="14" description="Product Features" url="http://cdn.cnetcontent.com/f7/d6/f7d61727-a2da-4add-8444-97d5a1d2c989.xml" />
				  </docCollection>
				</cnet>
====================================================================================================
*/
IF  EXISTS (SELECT * FROM sys.objects WHERE name = 'CCS_sp_get_DC2XML') DROP PROCEDURE CCS_sp_get_DC2XML
GO

CREATE PROCEDURE CCS_sp_get_DC2XML
	 @ProdID AS VARCHAR(40)
	,@LangCode AS VARCHAR(20)
AS BEGIN
/* BEGIN PROCEDURE */


DECLARE @images XML
DECLARE @xml as XML
DECLARE @doc as XML
DECLARE @imageType VARCHAR(200)


-- CREATE TABLE OF DIGITAL CONTENT
DECLARE @ProductContent TABLE
(
	 ContentGuid UNIQUEIDENTIFIER
	,MimeType VARCHAR(50)
	,URL VARCHAR(500)
	,MediaTypeID SMALLINT
	,ImageType VARCHAR(200)
	,ImageWeight SMALLINT
	,ImageId UNIQUEIDENTIFIER
	,ImageAngle VARCHAR(200)
)

INSERT @ProductContent SELECT
	 content.ContentGuid
	,content.MimeType
	,content.URL
	,content.MediaTypeID
	,MAX(CASE WHEN meta.MetaAtrId = 2 THEN v.MetaValueName ELSE NULL END) AS ImageType
	,MAX(CASE WHEN meta.MetaAtrId = 7 THEN v.MetaValueName ELSE NULL END) AS ImageWeight
	,MAX(CASE WHEN meta.MetaAtrId = 8 THEN v.MetaValueName ELSE NULL END) AS ImageId
	,MAX(CASE WHEN meta.MetaAtrId = 1 THEN v.MetaValueName ELSE NULL END) AS ImageAngle
		
FROM cds_DigContent_Links l
JOIN cds_DigContent content ON content.ContentGuid = l.ContentGuid
JOIN cds_DigContent_Lang_Links lang ON lang.ContentGuid = content.ContentGuid
LEFT OUTER JOIN cds_DigContent_Meta meta ON content.ContentGuid = meta.ContentGuid
LEFT OUTER JOIN cds_DigContent_Meta_ValVoc v ON v.MetaValueId = meta.MetaValueId
	
WHERE
	l.ProdID = '4452711'
	AND (	
		lang.LanguageCode = @LangCode
		OR lang.LanguageCode = 'Inv')
	
GROUP BY
	 content.ContentGuid
	,content.MimeType
	,content.URL
	,content.MediaTypeID
	
ORDER BY
	ImageWeight DESC


-- CREATE XML FOR IMAGE CONTENT
SET @images =
(
	SELECT -- <imageCollection>
		 pc.ImageId AS [@image-id]
		,15 AS [@media-type]
		,pc.ImageAngle AS [@angle]
		,(
			SELECT -- <image>
				 pci.MimeType AS [@mime-type]
				,pci.URL AS [@url]
				,(
					SELECT DISTINCT -- <attribute>
						a.MetaAtrId [@attrId]
						,a.MetaAtrName  [@name]
						,v.MetaValueId AS [@value]
					FROM cds_DigContent_Meta m
					JOIN cds_DigContent_Meta_AtrVoc a ON a.MetaAtrId = m.MetaAtrId
					JOIN cds_DigContent_Meta_ValVoc v ON v.MetaValueId = m.MetaValueId
					WHERE m.ContentGuid = pci.ContentGuid
					FOR XML PATH('attribute'), TYPE)
			FROM
				@ProductContent pci
			WHERE
				pci.ImageId = pc.ImageId
			GROUP BY	
				pci.ContentGuid, pci.MimeType, pci.URL
			FOR XML PATH('image'), TYPE)		
	
	FROM @ProductContent pc

	WHERE pc.MediaTypeID = 15

	GROUP BY
		 pc.ImageId
		,pc.ImageAngle
		
	FOR XML PATH('imageCollection'), ROOT('images'), TYPE
)

-- INSERT ATTRIBUTE imageType INTO ELEMENT <images>
SELECT TOP 1 @imageType = ImageType FROM @ProductContent
SET @images.modify('insert attribute type {sql:variable("@imageType")} into /images[1]')


-- CREATE XML FOR XML CONTENT
SET @xml =
(	
	SELECT
		 pc.MediaTypeID '@type'
        ,t.MediaTypeDescription '@description'
        ,pc.url '@url'
	
	FROM @ProductContent pc	
	JOIN cds_DigContent_Media_Types t ON t.MediaTypeID = pc.MediaTypeID
	
	WHERE pc.MediaTypeID IN (4,5,10)
	
	ORDER BY pc.MediaTypeID
	
	FOR XML PATH('xml'), ROOT('xmlCollection'), TYPE
)  

-- CREATE XML FOR DOC CONTENT
SET @doc = 
(
	SELECT -- <doc>
		 pc.MediaTypeID '@type'
		,t.MediaTypeDescription '@description'
		,pc.url '@url'
		,(
			SELECT -- <attribute>
				 a.MetaAtrId [@attrId]
				,a.MetaAtrName  [@name]
				,v.MetaValueId AS [@value]
				
			FROM cds_DigContent_Meta m
			JOIN cds_DigContent_Meta_AtrVoc a ON a.MetaAtrId = m.MetaAtrId
			JOIN cds_DigContent_Meta_ValVoc v ON v.MetaValueId = m.MetaValueId
			
			WHERE m.ContentGuid = pc.ContentGuid
			
			FOR XML PATH('attribute'), TYPE)
			
	FROM @ProductContent pc
	JOIN cds_DigContent_Media_Types t ON t.MediaTypeID = pc.MediaTypeID
	
	WHERE pc.MediaTypeID NOT IN (15, 4,5,10)
	
	ORDER BY pc.MediaTypeID
	
	FOR XML PATH('doc'), ROOT('docCollection'), TYPE
)  


SELECT @images, @xml, @doc
FOR XML PATH('cnet'), TYPE


/* ENDOF PROCEDURE */
END
GO

EXEC CCS_sp_get_DC2XML '4452711', 'en'